<?php
include_once '../baseInfo.php';
include_once '../config.php';

$now   = time();
$limit = 500;

// دیگر از OFFSET استفاده نمی‌کنیم
@unlink('warnOffset.txt');

/* -------------------- Helpers -------------------- */
function normalizeTs($ts) {
    if ($ts === null || $ts === '' || !is_numeric($ts)) return null;
    $ts = (int)$ts;
    if ($ts > 2000000000) $ts = intdiv($ts, 1000); // ms->s
    return $ts > 0 ? $ts : null;
}

// امن‌سازی برای متن ساده (نه Markdown/HTML) — اگر sendMessage شما Markdown/HTML می‌فرستد، به تناسب تغییر بده
function safeName($s) {
    $s = trim((string)$s);
    // حذف کاراکترهای کنترل و نرمال‌سازی فاصله‌ها
    $s = preg_replace('/[\x00-\x1F\x7F]/u', '', $s);
    $s = preg_replace('/\s+/u', ' ', $s);
    return $s;
}

/* -------------------- LOOP 1: سفارش‌های موعددار (هشدار/اعلان فوری) -------------------- */
// رکوردهایی که باید الان بررسی شوند: notif=0 (تا الان هشداری نگرفته) یا notif < now (وقتش رسیده)
$stmt = $connection->prepare("
    SELECT * FROM `orders_list`
    WHERE `status`=1 AND (`notif`=0 OR `notif` < ?)
    ORDER BY
      (notif = 0) DESC,
      CASE WHEN notif=0 THEN 0 ELSE notif END ASC,
      id ASC
    LIMIT ?
");
$stmt->bind_param("ii", $now, $limit);
$stmt->execute();
$orders = $stmt->get_result();
$stmt->close();

if ($orders && $orders->num_rows > 0) {
  while ($order = $orders->fetch_assoc()) {

    $orderId   = (int)$order['id'];
    $from_id   = $order['userid'];
    $remark    = $order['remark'] ?? '';
    $uuid      = $order['uuid'] ?? "0";
    $server_id = (int)$order['server_id'];
    $inbound_id= (int)$order['inbound_id'];
    $notif     = (int)$order['notif'];

    // نام قابل نمایش: اول remark، بعد ایمیلِ کلاینت match‌شده، بعد UUID
    $displayName = safeName($remark);
    $matchedEmail = '';

    // کانفیگ سرور
    $stmt = $connection->prepare("SELECT * FROM `server_config` WHERE `id`=?");
    $stmt->bind_param('i', $server_id);
    $stmt->execute();
    $serverConfig = $stmt->get_result()->fetch_assoc();
    $stmt->close();

    $serverType = $serverConfig['type'] ?? '';

    // مقادیر مشترک
    $found=false; $logedIn=false;
    $total=null;  $up=0; $down=0; $enable=true; $expiryTime=null;

    if ($serverType === "marzban") {
        // --- Marzban ---
        $info = getMarzbanUser($server_id, $remark); // معمولاً remark = username
        if (isset($info->username)) {
            $found   = true;
            $logedIn = true;
            $total   = isset($info->data_limit) ? (int)$info->data_limit : null; // bytes یا null
            $up      = isset($info->used_traffic) ? (int)$info->used_traffic : 0;
            $down    = 0; // مجموع مصرف در used_traffic
            $expiryTime = $info->expire ?? null; // ممکنه ms باشد
            $enable = (($info->status ?? '') === 'active');
            if ($displayName === '') $displayName = $info->username; // fallback
        } elseif (isset($info->detail) && $info->detail === "User not found") {
            $logedIn = true;
        }
    } else {
        // --- x-ui / sanaei ---
        $response = getJson($server_id);
        if ($response && $response->success) {
            $logedIn = true;
            $response = $response->obj;
            foreach ($response as $row) {
                if ($inbound_id == 0) {
                    // همه‌ی کلاینت‌ها را بگرد
                    $clients = [];
                    $settingsObj = json_decode($row->settings ?? '{}');
                    if (isset($settingsObj->clients) && is_array($settingsObj->clients)) {
                        $clients = $settingsObj->clients;
                    }
                    foreach ($clients as $c) {
                        $cid = $c->id ?? null;
                        $cpw = $c->password ?? null;
                        if (($cid && $cid == $uuid) || ($cpw && $cpw == $uuid)) {
                            $found = true;
                            $total = isset($row->total) ? (int)$row->total : null;
                            $up    = isset($row->up)   ? (int)$row->up   : 0;
                            $down  = isset($row->down) ? (int)$row->down : 0;
                            $expiryTime = $row->expiryTime ?? null;
                            $enable = isset($row->enable) ? (bool)$row->enable : true;
                            // اگر remark خالی بود، ایمیل کلاینت match‌شده را بعنوان نام نشان بده
                            $matchedEmail = safeName($c->email ?? '');
                            break 2;
                        }
                    }
                } else {
                    if ((int)$row->id === $inbound_id) {
                        $settings = json_decode($row->settings ?? '[]', true);
                        $clients  = $settings['clients'] ?? [];
                        $clientsStates = $row->clientStats ?? [];
                        $statesByEmail = [];
                        foreach ($clientsStates as $st) {
                            $statesByEmail[$st->email ?? ''] = $st;
                        }
                        foreach ($clients as $client) {
                            $cid = $client['id'] ?? null;
                            $cpw = $client['password'] ?? null;
                            if (($cid && $cid == $uuid) || ($cpw && $cpw == $uuid)) {
                                $found = true;
                                $email = $client['email'] ?? '';
                                $st    = $statesByEmail[$email] ?? null;
                                $total = isset($client['totalGB']) ? (int)$client['totalGB'] : null;
                                $up    = isset($st->up)    ? (int)$st->up    : 0;
                                $down  = isset($st->down)  ? (int)$st->down  : 0;
                                $enable= isset($st->enable)? (bool)$st->enable: true;
                                $expiryTime = $st->expiryTime ?? null;
                                $matchedEmail = safeName($email);
                                break 2;
                            }
                        }
                    }
                }
            }
        }
        // اگر از پنل زمان نگرفتیم ولی در DB expire_date هست، استفاده کن
        if (!$expiryTime && !empty($order['expire_date'])) {
            $expiryTime = $order['expire_date'];
        }
    }

    if (!$logedIn || !$found) {
        // رکورد پیدا نشد/لاگ‌این نشد — ردش کن
        continue;
    }

    // آماده‌سازی نام قابل نمایش
    if ($displayName === '') $displayName = $matchedEmail;
    if ($displayName === '' && !empty($uuid)) $displayName = $uuid;
    $displayName = safeName($displayName);

    // محاسبات مشترک
    $expiryTimeSec = normalizeTs($expiryTime);
    $leftgb = null;
    if ($total !== null) {
        $used = (int)$up + (int)$down;
        $totalLeft = max(0, (int)$total - $used);
        $leftgb = round($totalLeft / 1073741824, 2);
    }

    /* -------- پیام فوری (اتمام/غیرفعال/گذشتن زمان) با آپدیت اتمیک -------- */
    if ($notif == 0 && (
        ($leftgb !== null && $leftgb <= 0) ||
        ($expiryTimeSec !== null && $expiryTimeSec <= $now) ||
        (!$enable)
    )) {
        // اول notif را اتمیک جلو می‌بریم؛ اگر موفق بود، پیام می‌فرستیم
        $newTime = $now + 2*86400; // دو روز بعد برای حذف
        $stmt = $connection->prepare("UPDATE `orders_list` SET `notif`=? WHERE `id`=? AND `notif`=0");
        $stmt->bind_param("ii", $newTime, $orderId);
        $stmt->execute();
        $rows = $stmt->affected_rows;
        $stmt->close();

        if ($rows > 0) {
            $msg = "💡 کاربر گرامی،
اشتراک «{$displayName}» به پایان رسیده/غیرفعال شده است. می‌توانید از «خریدهای من» تمدید کنید یا سرویس جدید بخرید.";
            sendMessage($msg, null, null, $from_id);
        }
        continue;
    }

    /* -------- هشدار نزدیک انقضا (<1 روز) یا کمتر از 1GB -------- */
    if ($notif == 0) {
        $shouldWarnTime = ($expiryTimeSec !== null && $expiryTimeSec < $now + 86400);
        $shouldWarnData = ($leftgb !== null && $leftgb < 1);
        if ($shouldWarnTime || $shouldWarnData) {

            // ابتدا notif را اتمیک به آینده ببریم؛ سپس پیام
            $nextCheck = $now + 3600; // یک ساعت دیگر
            if ($expiryTimeSec !== null) $nextCheck = min($nextCheck, $expiryTimeSec);

            $stmt = $connection->prepare("UPDATE `orders_list` SET `notif`=? WHERE `id`=? AND `notif`=0");
            $stmt->bind_param("ii", $nextCheck, $orderId);
            $stmt->execute();
            $rows = $stmt->affected_rows;
            $stmt->close();

            if ($rows > 0) {
                $unit = $shouldWarnTime ? "روز" : "گیگ";
                $msg = "💡 کاربر گرامی،
از سرویس اشتراک «{$displayName}» تنها (۱ {$unit}) باقی مانده است. می‌توانید از «خریدهای من» تمدید کنید یا سرویس جدید خریداری کنید.";
                sendMessage($msg, null, null, $from_id);
            }
            continue;
        }
    }

    // اگر اکانت غیرفعال است و قبلاً notif ست نشده بود
    if (!$enable && $notif <= 0) {
        $newTime = $now + 2*86400;
        $stmt = $connection->prepare("UPDATE `orders_list` SET `notif`=? WHERE `id`=? AND `notif`<=0");
        $stmt->bind_param("ii", $newTime, $orderId);
        $stmt->execute();
        $stmt->close();
        continue;
    }
  }
}

/* -------------------- LOOP 2: حذف نهایی پس از موعد -------------------- */
$stmt = $connection->prepare("
    SELECT * FROM `orders_list`
    WHERE `status`=1 AND `notif` > 0 AND `notif` < ?
    LIMIT 200
");
$stmt->bind_param("i", $now);
$stmt->execute();
$orders = $stmt->get_result();
$stmt->close();

if ($orders && $orders->num_rows > 0) {
  while ($order = $orders->fetch_assoc()) {
    $orderId   = (int)$order['id'];
    $from_id   = $order['userid'];
    $remark    = $order['remark'] ?? '';
    $uuid      = $order['uuid'] ?? "0";
    $server_id = (int)$order['server_id'];
    $inbound_id= (int)$order['inbound_id'];

    // نام برای پیام حذف
    $displayName = safeName($remark);
    if ($displayName === '' && !empty($uuid)) $displayName = $uuid;

    // نوع پنل
    $stmt = $connection->prepare("SELECT `type` FROM `server_config` WHERE `id`=?");
    $stmt->bind_param('i', $server_id);
    $stmt->execute();
    $rowConf = $stmt->get_result()->fetch_assoc();
    $stmt->close();
    $serverType = $rowConf['type'] ?? '';

    // حذف از پنل
    if ($serverType === "marzban") {
        $res = deleteMarzban($server_id, $remark);
    } else {
        if ($inbound_id > 0) $res = deleteClient($server_id, $inbound_id, $uuid, 1);
        else $res = deleteInbound($server_id, $uuid, 1);
    }

    if (!is_null($res)) {
        // پیام حذف
        $msg = "💡 کاربر گرامی،
اشتراک «{$displayName}» منقضی شد و از لیست سفارش‌ها حذف گردید. لطفاً از فروشگاه سرویس جدید خریداری کنید.";
        sendMessage($msg, null, null, $from_id);

        // حذف رکورد از DB
        $stmt = $connection->prepare("DELETE FROM `orders_list` WHERE `uuid`=?");
        $stmt->bind_param("s", $uuid);
        $stmt->execute();
        $stmt->close();
    } else {
        // در صورت شکست حذف، ۶ ساعت بعد دوباره تلاش کن
        $retry = $now + 6*3600;
        $stmt = $connection->prepare("UPDATE `orders_list` SET `notif`=? WHERE `id`=?");
        $stmt->bind_param("ii", $retry, $orderId);
        $stmt->execute();
        $stmt->close();
    }
  }
}
